<?php

declare(strict_types=1);

namespace App\Http\Controllers\Admin\Masters;

use App\Http\Controllers\Admin\Controller;
use App\Http\Requests\Admin\Masters\BrandStoreRequest;
use App\Http\Requests\Admin\Masters\BrandUpdateRequest;
use App\Models\Brand;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class BrandController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $brands = Brand::all();

        return view('admin.masters.brands', compact('brands'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): void
    {
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(BrandStoreRequest $request)
    {
        try
        {
            DB::beginTransaction();
            $input = $request->validated();
            if (isset($input['logo']))
            {
                $input['logo'] = 'storage/'.$request->file('logo')->store('brands', 'public');
            }
            Brand::create(Arr::only($input, Brand::getFillables()));
            DB::commit();

            return response()->json(['success' => 'Brand created successfully!']);
        }
        catch (Exception $e)
        {
            DB::rollBack();

            return $this->respondWithAjax($e, 'creating', 'Brand');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id): void
    {
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Brand $brand)
    {
        $brandLogoHtml = '<a href="'.asset($brand->logo).'" target="_blank">
                                <div class="card p-0 m-0">
                                    <div class="card-body p-0 m-0">
                                        <img src="'.asset($brand->logo).'" class="img-fluid" alt="brand_logo">
                                    </div>
                                </div>
                            </a>';

        return [
            'result'        => 1,
            'brand'         => $brand,
            'brandLogoHtml' => $brandLogoHtml,
        ];
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(BrandUpdateRequest $request, Brand $brand)
    {
        try
        {
            DB::beginTransaction();
            $input = $request->validated();
            if (isset($input['logo']))
            {
                $input['logo'] = 'storage/'.$request->file('logo')->store('brands', 'public');
                if (File::exists(public_path($brand->logo)))
                {
                    File::delete(public_path($brand->logo));
                }
            }
            else
            {
                $input['logo'] = $brand->logo;
            }

            $brand->update(Arr::only($input, Brand::getFillables()));
            DB::commit();

            return response()->json(['success' => 'Brand updated successfully!']);
        }
        catch (Exception $e)
        {
            DB::rollBack();

            return $this->respondWithAjax($e, 'creating', 'Brand');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Brand $brand)
    {
        try
        {
            DB::beginTransaction();
            if (File::exists(public_path('\\').$brand->logo))
            {
                File::delete(public_path('\\').$brand->logo);
            }
            $brand->delete();
            DB::commit();

            return response()->json(['success' => 'Brand deleted successfully!']);
        }
        catch (Exception $e)
        {
            return $this->respondWithAjax($e, 'deleting', 'Brand');
        }
    }
}
